import Foundation
import HealthKit
import XCTest

/// Mock data providers for comprehensive testing
/// Provides realistic test data across all app modules
class MockDataProviders {
    
    // MARK: - Actuarial Test Data
    struct MockActuarialData {
        static let baselineScenarios = [
            ActuarialScenario(
                age: 25,
                gender: .male,
                baseLifeExpectancy: 76.1,
                expectedRiskFactors: ["smoking": 0.0, "drinking": 0.1]
            ),
            ActuarialScenario(
                age: 35,
                gender: .female,
                baseLifeExpectancy: 81.2,
                expectedRiskFactors: ["smoking": 0.0, "drinking": 0.05]
            ),
            ActuarialScenario(
                age: 45,
                gender: .male,
                baseLifeExpectancy: 76.1,
                expectedRiskFactors: ["smoking": 0.2, "drinking": 0.15]
            ),
            ActuarialScenario(
                age: 65,
                gender: .female,
                baseLifeExpectancy: 81.2,
                expectedRiskFactors: ["smoking": 0.0, "drinking": 0.0]
            )
        ]
        
        static let edgeCaseScenarios = [
            ActuarialScenario(
                age: 18,  // Minimum age
                gender: .male,
                baseLifeExpectancy: 76.1,
                expectedRiskFactors: [:]
            ),
            ActuarialScenario(
                age: 100, // Maximum realistic age
                gender: .female,
                baseLifeExpectancy: 81.2,
                expectedRiskFactors: [:]
            )
        ]
    }
    
    // MARK: - HealthKit Mock Data
    class MockHealthKitService {
        static let sampleHealthData = [
            HKQuantityTypeIdentifier.stepCount: [
                MockHealthSample(value: 8500, date: Date().addingTimeInterval(-86400)),
                MockHealthSample(value: 10200, date: Date().addingTimeInterval(-172800)),
                MockHealthSample(value: 7800, date: Date().addingTimeInterval(-259200))
            ],
            HKQuantityTypeIdentifier.heartRate: [
                MockHealthSample(value: 72, date: Date().addingTimeInterval(-3600)),
                MockHealthSample(value: 68, date: Date().addingTimeInterval(-7200)),
                MockHealthSample(value: 75, date: Date().addingTimeInterval(-10800))
            ],
            HKQuantityTypeIdentifier.activeEnergyBurned: [
                MockHealthSample(value: 450, date: Date().addingTimeInterval(-86400)),
                MockHealthSample(value: 520, date: Date().addingTimeInterval(-172800))
            ]
        ]
        
        static func simulateAuthorizationFlow() -> MockAuthorizationResult {
            return MockAuthorizationResult(
                status: .sharingAuthorized,
                requestedTypes: [
                    HKQuantityType.quantityType(forIdentifier: .stepCount)!,
                    HKQuantityType.quantityType(forIdentifier: .heartRate)!,
                    HKQuantityType.quantityType(forIdentifier: .activeEnergyBurned)!
                ]
            )
        }
        
        static func simulateDataFetch(for type: HKQuantityTypeIdentifier) -> [MockHealthSample] {
            return sampleHealthData[type] ?? []
        }
        
        static func simulateBackgroundDelivery() -> MockBackgroundDelivery {
            return MockBackgroundDelivery(
                frequency: HKUpdateFrequency.immediate,
                lastDelivery: Date().addingTimeInterval(-1800),
                sampleCount: 3
            )
        }
    }
    
    // MARK: - UI Test Data
    struct MockUIData {
        static let countdownFormats = [
            CountdownFormat(
                totalSeconds: 1_892_160_000, // ~60 years
                displayFormat: "60y 0m 0d 0h 0m 0s",
                shortFormat: "60 years"
            ),
            CountdownFormat(
                totalSeconds: 86400, // 1 day
                displayFormat: "0y 0m 1d 0h 0m 0s",
                shortFormat: "1 day"
            ),
            CountdownFormat(
                totalSeconds: 3661, // 1 hour, 1 minute, 1 second
                displayFormat: "0y 0m 0d 1h 1m 1s",
                shortFormat: "1h 1m"
            )
        ]
        
        static let visualIndicators = [
            VisualIndicator(
                remainingYears: 60,
                colorScheme: .green,
                urgencyLevel: .low
            ),
            VisualIndicator(
                remainingYears: 30,
                colorScheme: .yellow,
                urgencyLevel: .medium
            ),
            VisualIndicator(
                remainingYears: 10,
                colorScheme: .red,
                urgencyLevel: .high
            )
        ]
    }
    
    // MARK: - Privacy Test Data
    struct MockPrivacyData {
        static let userDataSample = UserDataSample(
            predictions: [
                PredictionRecord(date: Date(), lifeExpectancy: 76.5, riskFactors: ["smoking": 0.1]),
                PredictionRecord(date: Date().addingTimeInterval(-86400), lifeExpectancy: 76.8, riskFactors: [:])
            ],
            healthData: [
                "steps": "85000",
                "heartRate": "72 BPM",
                "activeCalories": "2500 cal"
            ],
            preferences: [
                "notifications": "enabled",
                "dataRetention": "1 year",
                "shareData": "disabled"
            ]
        )
        
        static let deletionScenarios = [
            DeletionScenario(
                dataType: .allUserData,
                expectedDuration: 0.5, // seconds
                verificationMethod: .databaseQuery
            ),
            DeletionScenario(
                dataType: .predictionHistory,
                expectedDuration: 0.2,
                verificationMethod: .fileSystemCheck
            ),
            DeletionScenario(
                dataType: .userPreferences,
                expectedDuration: 0.1,
                verificationMethod: .userDefaultsCheck
            )
        ]
    }
    
    // MARK: - Performance Test Data
    struct MockPerformanceData {
        static let calculationBenchmarks = [
            PerformanceBenchmark(
                operation: "Basic Life Expectancy Calculation",
                expectedDuration: 0.001, // 1ms
                maxDuration: 0.005, // 5ms
                inputComplexity: .low
            ),
            PerformanceBenchmark(
                operation: "Complex Risk Factor Analysis",
                expectedDuration: 0.010, // 10ms
                maxDuration: 0.050, // 50ms
                inputComplexity: .high
            ),
            PerformanceBenchmark(
                operation: "Database Query - Prediction History",
                expectedDuration: 0.020, // 20ms
                maxDuration: 0.100, // 100ms
                inputComplexity: .medium
            )
        ]
        
        static let memoryBenchmarks = [
            MemoryBenchmark(
                operation: "App Launch",
                expectedMemoryMB: 25,
                maxMemoryMB: 50
            ),
            MemoryBenchmark(
                operation: "Background Calculation",
                expectedMemoryMB: 15,
                maxMemoryMB: 30
            ),
            MemoryBenchmark(
                operation: "Data Export",
                expectedMemoryMB: 35,
                maxMemoryMB: 70
            )
        ]
    }
}

// MARK: - Supporting Mock Structures
struct ActuarialScenario {
    let age: Int
    let gender: Gender
    let baseLifeExpectancy: Double
    let expectedRiskFactors: [String: Double]
}

enum Gender {
    case male, female, other
}

struct MockHealthSample {
    let value: Double
    let date: Date
}

struct MockAuthorizationResult {
    let status: HKAuthorizationStatus
    let requestedTypes: [HKQuantityType]
}

struct MockBackgroundDelivery {
    let frequency: HKUpdateFrequency
    let lastDelivery: Date
    let sampleCount: Int
}

struct CountdownFormat {
    let totalSeconds: TimeInterval
    let displayFormat: String
    let shortFormat: String
}

struct VisualIndicator {
    let remainingYears: Int
    let colorScheme: ColorScheme
    let urgencyLevel: UrgencyLevel
}

enum ColorScheme {
    case green, yellow, red
}

enum UrgencyLevel {
    case low, medium, high
}

struct UserDataSample {
    let predictions: [PredictionRecord]
    let healthData: [String: String]
    let preferences: [String: String]
}

struct PredictionRecord {
    let date: Date
    let lifeExpectancy: Double
    let riskFactors: [String: Double]
}

struct DeletionScenario {
    let dataType: DataType
    let expectedDuration: TimeInterval
    let verificationMethod: VerificationMethod
}

enum DataType {
    case allUserData, predictionHistory, userPreferences
}

enum VerificationMethod {
    case databaseQuery, fileSystemCheck, userDefaultsCheck
}

struct PerformanceBenchmark {
    let operation: String
    let expectedDuration: TimeInterval
    let maxDuration: TimeInterval
    let inputComplexity: InputComplexity
}

enum InputComplexity {
    case low, medium, high
}

struct MemoryBenchmark {
    let operation: String
    let expectedMemoryMB: Int
    let maxMemoryMB: Int
}